<?php

namespace app\models;

use Yii;
use yii\base\Model;

/**
 * Register form
 */
class RegisterForm extends Model
{
    public $full_name;
    public $email;
    public $phone;
    public $password;
    public $password_repeat;
    public $agree;

    private $_user = false;

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['full_name', 'email', 'phone', 'password', 'password_repeat', 'agree'], 'required'],
            
            ['full_name', 'trim'],
            ['full_name', 'match', 'pattern' => '/^[а-яА-ЯёЁ\s\-]+$/u', 'message' => 'ФИО должно содержать только русские буквы, пробелы и дефисы'],
            ['full_name', 'string', 'min' => 2, 'max' => 255],
            
            ['email', 'trim'],
            ['email', 'email', 'message' => 'Введите корректный email адрес'],
            ['email', 'string', 'max' => 100],
            ['email', 'unique', 'targetClass' => User::class, 'message' => 'Этот email уже занят.'],
            
            ['phone', 'trim'],
            ['phone', 'match', 'pattern' => '/^\+7\(\d{3}\)\d{3}-\d{2}-\d{2}$/', 'message' => 'Формат телефона: +7(XXX)XXX-XX-XX'],
            
            ['password', 'string', 'min' => 7, 'message' => 'Пароль должен содержать минимум 7 символов'],
            ['password_repeat', 'compare', 'compareAttribute' => 'password', 'message' => 'Пароли не совпадают'],
            
            ['agree', 'required', 'requiredValue' => 1, 'message' => 'Вы должны согласиться с правилами регистрации'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'full_name' => 'ФИО',
            'email' => 'Email',
            'phone' => 'Телефон',
            'password' => 'Пароль',
            'password_repeat' => 'Повторите пароль',
            'agree' => 'Согласен с правилами регистрации',
        ];
    }

    /**
     * Регистрирует пользователя
     * @return bool whether the user is registered successfully
     */
    public function register()
    {
        if (!$this->validate()) {
            return false;
        }

        $user = new User();
        $user->full_name = $this->full_name;
        $user->email = $this->email;
        $user->phone = $this->phone;
        $user->setPassword($this->password);
        $user->generateAuthKey();
        $user->status = User::STATUS_ACTIVE;
        $user->role = User::ROLE_EMPLOYEE;

        if ($user->save()) {
            // Отправляем email с подтверждением (можно добавить позже)
            // $this->sendEmail($user);
            
            return true;
        }

        return false;
    }

    /**
     * Sends confirmation email to user
     * @param User $user user model to with email should be send
     * @return bool whether the email was sent
     */
    protected function sendEmail($user)
    {
        return Yii::$app
            ->mailer
            ->compose(
                ['html' => 'emailVerify-html', 'text' => 'emailVerify-text'],
                ['user' => $user]
            )
            ->setFrom([Yii::$app->params['supportEmail'] => Yii::$app->name . ' robot'])
            ->setTo($this->email)
            ->setSubject('Регистрация аккаунта в ' . Yii::$app->name)
            ->send();
    }
}