<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\data\ActiveDataProvider;
use yii\web\NotFoundHttpException;
use app\models\EmployeeProfile;
use app\models\User;

class AdminController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            return Yii::$app->user->identity->isAdmin();
                        }
                    ],
                ],
                'denyCallback' => function ($rule, $action) {
                    return $this->redirect(['/site/index']);
                }
            ],
        ];
    }

    /**
     * Главная страница администратора
     */
    public function actionIndex()
    {
        $query = EmployeeProfile::find()->joinWith('user');
        
        // Применяем фильтры
        $filters = Yii::$app->request->get();
        if (!empty($filters)) {
            foreach ($filters as $field => $value) {
                if (!empty($value) && in_array($field, ['status', 'education', 'gender', 'position'])) {
                    $query->andWhere([$field => $value]);
                }
            }
        }
        
        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 10,
            ],
            'sort' => [
                'defaultOrder' => ['created_at' => SORT_DESC],
                'attributes' => [
                    'created_at',
                    'status',
                    'full_name' => [
                        'asc' => ['user.full_name' => SORT_ASC],
                        'desc' => ['user.full_name' => SORT_DESC],
                    ],
                    'position',
                    'experience',
                ],
            ],
        ]);

        // Статистика
        $totalProfiles = EmployeeProfile::find()->count();
        $newProfiles = EmployeeProfile::find()->where(['status' => 'new'])->count();
        $reviewProfiles = EmployeeProfile::find()->where(['status' => 'in_review'])->count();
        $acceptedProfiles = EmployeeProfile::find()->where(['status' => 'accepted'])->count();

        return $this->render('index', [
            'dataProvider' => $dataProvider,
            'totalProfiles' => $totalProfiles,
            'newProfiles' => $newProfiles,
            'reviewProfiles' => $reviewProfiles,
            'acceptedProfiles' => $acceptedProfiles,
        ]);
    }

    /**
     * Просмотр анкеты администратором
     */
    public function actionView($id)
    {
        $model = $this->findModel($id);

        return $this->render('view', [
            'model' => $model,
        ]);
    }

    /**
     * Изменение статуса анкеты
     */
    public function actionUpdateStatus($id, $status)
    {
        $model = $this->findModel($id);
        
        if (Yii::$app->request->isPost) {
            $comment = Yii::$app->request->post('comment');
            
            if (in_array($status, [EmployeeProfile::STATUS_IN_REVIEW, EmployeeProfile::STATUS_ACCEPTED])) {
                $model->status = $status;
                $model->reviewed_by = Yii::$app->user->id;
                $model->reviewed_at = date('Y-m-d H:i:s');
                $model->review_comment = $comment;
                
                if ($model->save()) {
                    Yii::$app->session->setFlash('success', 
                        '✅ Статус анкеты успешно обновлен.'
                    );
                } else {
                    Yii::$app->session->setFlash('error', 
                        'Ошибка при обновлении статуса.'
                    );
                }
            }
            
            return $this->redirect(['view', 'id' => $id]);
        }

        return $this->render('update-status', [
            'model' => $model,
            'status' => $status,
        ]);
    }

    /**
     * Экспорт данных
     */
    public function actionExport($format = 'csv')
    {
        $profiles = EmployeeProfile::find()->joinWith('user')->all();
        
        if ($format === 'csv') {
            return $this->exportToCsv($profiles);
        }
        
        return $this->redirect(['index']);
    }

    /**
     * Экспорт в CSV
     */
    private function exportToCsv($profiles)
    {
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=employee_profiles_' . date('Y-m-d') . '.csv');
        
        $output = fopen('php://output', 'w');
        
        // Заголовки
        fputcsv($output, [
            'ФИО',
            'Email',
            'Телефон',
            'Должность',
            'Статус анкеты',
            'Дата заполнения',
            'Образование',
            'Стаж (месяцев)',
            'Гражданство',
            'Пол',
            'Семейное положение',
        ], ';');
        
        // Данные
        foreach ($profiles as $profile) {
            fputcsv($output, [
                $profile->user->full_name,
                $profile->user->email,
                $profile->user->phone,
                $profile->position,
                $profile->getStatusName(),
                $profile->created_at,
                $profile->getEducationName(),
                $profile->experience,
                $profile->citizenship,
                $profile->getGenderName(),
                $profile->getMaritalStatusName(),
            ], ';');
        }
        
        fclose($output);
        exit;
    }

    /**
     * Находит модель анкеты по ID
     */
    protected function findModel($id)
    {
        if (($model = EmployeeProfile::findOne($id)) !== null) {
            return $model;
        }

        throw new NotFoundHttpException('Анкета не найдена.');
    }
}